<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace PrestaShop\PrestaShop\Core\Context;

/**
 * This context service gives access to all contextual data related to employee.
 *
 * @experimental Depends on ADR https://github.com/PrestaShop/ADR/pull/36
 */
class EmployeeContext
{
    public const SUPER_ADMIN_PROFILE_ID = 1;

    public function __construct(
        protected readonly ?Employee $employee,
        protected readonly array $allShopsIds,
    ) {
    }

    public function getEmployee(): ?Employee
    {
        return $this->employee;
    }

    public function hasAuthorizationOnShopGroup(int $shopGroupId): bool
    {
        if (!$this->getEmployee()) {
            return false;
        }

        return $this->isSuperAdmin() || in_array($shopGroupId, $this->getEmployee()->getAssociatedShopGroupIds());
    }

    public function hasAuthorizationOnShop(int $shopId): bool
    {
        if (!$this->getEmployee()) {
            return false;
        }

        return $this->isSuperAdmin() || in_array($shopId, $this->getEmployee()->getAssociatedShopIds());
    }

    public function hasAuthorizationForAllShops(): bool
    {
        if (!$this->getEmployee()) {
            return false;
        }

        if ($this->isSuperAdmin()) {
            return true;
        }

        foreach ($this->allShopsIds as $shopId) {
            if (!$this->hasAuthorizationOnShop($shopId)) {
                return false;
            }
        }

        return true;
    }

    public function getDefaultShopId(): int
    {
        if (!$this->getEmployee()) {
            return 0;
        }

        return $this->getEmployee()->getDefaultShopId();
    }

    public function isSuperAdmin(): bool
    {
        return $this->getEmployee() && $this->getEmployee()->getProfileId() === self::SUPER_ADMIN_PROFILE_ID;
    }
}
